% countOverTime.m
%
% This is a script that I use to count pumping of the C. elegans
% pharynx.  It works like this:
% 
% > countOverTime
% > Name of strain: (N2)
% > Hit space bar to score worm 1's first condition...
% 
% After you hit space bar, the program will count down for as 
% many seconds specified, and then beep when it's finished.
% During this time, hit the space bar every time the worm pumps.
% you can score the light condition:
% 
% > Counted 25 pumps.  Hit enter to continue.
%
% This prompt is here so that spacebar
% presses after the elapsed time don't accidentally trigger the
% next condition (if there is one).
% 
% > Hit space bar to score worm 1's second condition...
% 
% This cycle will continue for all conditions, and then for the
% number of worms specified in the parameters below.  After this
% is all done, it will exclaim
%
% > Experiment complete! :)
%
% Then you can run it again to do another strain.
%
% The program will create a struct called 'strain' that will
% contain n fields, one for each condition.  It will save this
% struct to a file in your current directory (e.g. N2.mat).
%
% After all the data has been stored, run graphResults to 
% generate all sorts of fun graphs.  The graphs will compare
% conditions within strain (mean w/ SD) and across strains
% (mean w/o SD).

% PARAMETERS
% Edit the following before running the program to change various options.
nWorms = 20;         % Number of worms to score for the experiment.
nCond = 2;          % Number of conditions to score per animal (e.g. light & no-light)
nSec = 10;          % Number of seconds to score per condition
eventName = 'Pumps';% Use string of choice for "X per sec" in graph 
lg = {'No light'; 'Light'}; % Labels for graph legend
maxEvents = 80;     % Max number of events to record per condition
winWidth = 1000;    % Number of ms to use as a moving average in the graph
autoSave = true;    % Whether to automatically save the data to a file
genGraph = true;    % Graph will plot one line (mean +- SD) per condition
acquireData = true; % Turn to false if you just want to plot data
yMax = 5;

% CONSTANTS
ds = datestr(now, 'yyyymmdd');
numMsInS = 1000;

% Start
begin = input('Have you checked all the parameters in countOverTime.m? (y/n) ', 's');
if (begin == 'n')
    disp('Restart after you have confirmed all params!');
    return;
end

strain = input('Name of strain: ', 's');

if (acquireData)
    data = cell(nWorms, nCond);
    % Make my own for loop, so I can change the index within the loop
    w = 1;
    while(w <= nWorms)
        disp(['-Worm #' num2str(w) ':']); 
        % Make my own for loop, so I can change the index within the loop
        c = 1;
        while(c <= nCond)
            disp(['--Condition ' num2str(c) ':']);
            vec = zeros(maxEvents,1);

            k = 'n';
            while(true)
                k = input('  Press ''c'' then RETURN to start the timer, ''r'' to redo last trial:', 's');
                if (k == 'c')
                    break;
                elseif (k == 'r')
                    if (c == 1 && w ~= 1) % Started new worm - go back
                        c = nCond;
                        w = w - 1;
                    else
                        c = c - 1;
                    end
                    disp(['Redoing worm #' num2str(w) ', condition ' num2str(c)]);
                    break;
                end
            end

            t = timer('TimerFcn', ...
                      'timeup=true; beep;', ...
                      'StartDelay', nSec);
            start(t);
            tic

            timeup = false;
            nPresses = 0;
            while(timeup == false)
                pause;
                nPresses = nPresses + 1;
                vec(nPresses) = round(toc * numMsInS);  % Convert to ms for graphing
            end
            vec = vec(vec > 0);
            vec = vec(vec < nSec * numMsInS);  % Convert to ms
            data{w,c} = vec;

            disp(['Events counted: ', num2str(length(vec))]);
            
            % Give a chance to redo the very last trial
            if (w == nWorms && c == nCond)
                while(true)
                    k = input('Redo last trial of experiment? (y/n):', 's');
                    if (k == 'y')
                        c = c - 1;
                        break;
                    elseif (k == 'n')
                        break;
                    end
                end
            end
                
            c = c + 1;  % Manually increment counter, as matlab doesn't allow for indexes to change inside a loop
        end
        w = w + 1;
    end
end

varname = [strain '_' ds];
eval([varname ' = data;']);
if (autoSave && exist([varname '.mat'], 'file') == 0)
    save([varname '.mat'], varname);
else
    disp('NOTE: Data not automatically saved because autosave=false or file already exists.');
end

% Get y-vals for the plots
m = cell(nCond,1);
sd = cell(nCond,1);
for c=1:nCond
    [m{c} sd{c}] = eventsOverTime(data(:,c), nSec*numMsInS, winWidth, ...
        numMsInS, eventName, false);
end

if(genGraph)
    linecm = [24 79 245; 250 25 25] ./ 255;  % TODO: support more colors
    patchcm = [196 208 245; 250 200 200] ./ 255;
    f = figure;
    hold on
    x = (1/numMsInS):(1/numMsInS):nSec;
    for c=1:nCond
        p = patch([x x(end:-1:1)], [m{c}+sd{c} m{c}(end:-1:1)-sd{c}(end:-1:1)], ...
            patchcm(c,:), 'LineStyle', 'none');
    end
    h = zeros(nCond, 1);  % Vector of handles for legend
    for c=1:nCond
        h(c) = plot(x, m{c}, 'LineWidth', 3, 'Color', linecm(c,:));
    end

    t = title([eventName ' per sec (' varname ')'], 'FontWeight', 'bold');
    set(t, 'interpreter', 'none');
    xlabel('time (s)');
    ylabel([eventName ' per sec']);
    prettify();
    legend(h, lg);    
    a = axis;
    axis([winWidth/numMsInS nSec 0 yMax]);
    
    saveas(f, [varname '.fig']);
end